<?php
/**
 * Custom functions that act independently of the theme templates
 *
 * Eventually, some of the functionality here could be replaced by core features
 *
 * @package Snaps
 */

/**
 * Get our wp_nav_menu() fallback, wp_page_menu(), to show a home link.
 */
function snaps_page_menu_args( $args ) {
	$args['show_home'] = true;
	return $args;
}
add_filter( 'wp_page_menu_args', 'snaps_page_menu_args' );


/**
 * Adds custom classes to the array of body classes.
 */
function snaps_body_classes( $classes ) {
	// Adds a class of group-blog to blogs with more than 1 published author
	if ( is_multi_author() ) {
		$classes[] = 'group-blog';
	}

	return $classes;
}
add_filter( 'body_class', 'snaps_body_classes' );

/**
* Remove the 1st gallery shortcode from gallery post format content.
*/
function snaps_strip_first_gallery( $content ) {
	if ( ! is_single() && 'gallery' === get_post_format() && 'post' === get_post_type() && get_post_gallery() ) {
		$regex   = get_shortcode_regex( array( 'gallery' ) );
		$content = preg_replace( '/'. $regex .'/s', '', $content, 1 );
		$content = wp_kses_post( $content );
	}

	return $content;
}
add_filter( 'the_content', 'snaps_strip_first_gallery' );

if ( ! function_exists( 'snaps_continue_reading_link' ) ) :
/**
 * Returns an ellipsis and "Continue reading" plus off-screen title link for excerpts
 */
function snaps_continue_reading_link() {
	return '&hellip; <a href="'. esc_url( get_permalink() ) . '">' . sprintf( __( 'Continue reading <span class="screen-reader-text">%1$s</span>', 'snaps' ), esc_attr( strip_tags( get_the_title() ) ) ) . '</a>';
}
endif; // snaps_continue_reading_link


/**
 * Replaces "[...]" (appended to automatically generated excerpts) with snaps_continue_reading_link().
 *
 * To override this in a child theme, remove the filter and add your own
 * function tied to the excerpt_more filter hook.
 */
function snaps_auto_excerpt_more( $more ) {
	return snaps_continue_reading_link();
}
add_filter( 'excerpt_more', 'snaps_auto_excerpt_more' );


/**
 * Adds a pretty "Continue Reading" link to custom post excerpts.
 *
 * To override this link in a child theme, remove the filter and add your own
 * function tied to the get_the_excerpt filter hook.
 */
function snaps_custom_excerpt_more( $output ) {
	if ( has_excerpt() && ! is_attachment() ) {
		$output .= snaps_continue_reading_link();
	}
	return $output;
}
add_filter( 'get_the_excerpt', 'snaps_custom_excerpt_more' );
