<?php
/**
 * Snaps functions and definitions
 *
 * @package Snaps
 */

/**
 * Set the content width based on the theme's design and stylesheet.
 */
if ( ! isset( $content_width ) ) :
	$content_width = 1075; /* pixels */
endif;


if ( ! function_exists( 'snaps_setup' ) ) :
/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which runs
 * before the init hook. The init hook is too late for some features, such as indicating
 * support post thumbnails.
 */
function snaps_setup() {
	/**
	 * Make theme available for translation
	 * Translations can be filed in the /languages/ directory
	 * If you're building a theme based on snaps, use a find and replace
	 * to change 'snaps' to the name of your theme in all the template files
	 */
	load_theme_textdomain( 'snaps', get_template_directory() . '/languages' );

	/**
	 * Add default posts and comments RSS feed links to head
	 */
	add_theme_support( 'automatic-feed-links' );

	/**
	 * Let WordPress manage the document title.
	 * By adding theme support, we declare that this theme does not use a
	 * hard-coded <title> tag in the document head, and expect WordPress to
	 * provide it for us.
	 */
	add_theme_support( 'title-tag' );

	/**
	 * Add styles to WYSIWYG editor, so the text formatting will match the theme
	 */
	add_editor_style( array( 'editor-style.css', snaps_fonts_url(), 'genericons/genericons.css' ) );

	/**
	 * Register menus in theme
	 * This theme uses wp_nav_menu() in two locations
	 */
	register_nav_menus( array(
		'primary' => __( 'Primary Menu', 'snaps' ),
		'social' => __( 'Social Links', 'snaps' ),
	) );

	/*
	 * Switch default core markup for search form, comments, galleries and captions
	 * to output valid HTML5.
	 */
	add_theme_support( 'html5', array(
		'search-form',
		'comment-form',
		'comment-list',
		'gallery',
		'caption',
	) );

	/**
	 * Enable support for Post Thumbnails
	 * See http://codex.wordpress.org/Post_Formats
	 */
	add_theme_support( 'post-thumbnails' );

	/* setting custom image size for thumbnails for Snaps */
	add_image_size( 'snaps-thumbnails', '480', '640', true );

	/* setting custom image size for the Site Logo - double the max height set in CSS for retina screens */
	add_image_size( 'snaps-site-logo', '', '320', 9999 );

	/**
	 * Enable support for Post Formats
	 */
	add_theme_support( 'post-formats', array(
		'aside',
		'audio',
		'chat',
		'gallery',
		'image',
		'link',
		'quote',
		'status',
		'video',
	) );

	/**
	 * Set up the WordPress core custom background feature.
	 */
	add_theme_support( 'custom-background', apply_filters( 'snaps_custom_background_args', array(
		'default-color' => 'ffffff',
		'default-image' => '',
	) ) );
}
endif; // snaps_setup
add_action( 'after_setup_theme', 'snaps_setup' );


/**
 * Register Google Fonts
 */
function snaps_fonts_url() {
	$fonts_url = '';

	/* Translators: If there are characters in your language that are not
	 * supported by Lato, translate this to 'off'. Do not translate
	 * into your own language.
	 */
	$lato = _x( 'on', 'Lato font: on or off', 'minnow' );

	$font_families = array();

	if ( 'off' !== $lato ) {

		$font_families[] = 'Lato:700,900,700italic';

		$query_args = array(
			'family' => urlencode( implode( '|', $font_families ) ),
			'subset' => urlencode( 'latin,latin-ext' ),
		);

		$fonts_url = add_query_arg( $query_args, 'https://fonts.googleapis.com/css' );
	}

	return $fonts_url;
}


if ( ! function_exists( 'snaps_clearfix_class' ) ) :
/**
 * Add class to posts for clearfix
 */
function snaps_clearfix_class( $classes ) {
	$classes[] = 'clearfix';
	return $classes;
}
endif;
add_filter( 'post_class', 'snaps_clearfix_class', 10, 3 );


if ( ! function_exists( 'snaps_setup_author' ) ) :
/**
 * Sets the authordata global when viewing an author archive.
 *
 * This provides backwards compatibility with
 * http://core.trac.wordpress.org/changeset/25574
 *
 * It removes the need to call the_post() and rewind_posts() in an author
 * template to print information about the author.
 *
 * @global WP_Query $wp_query WordPress Query object.
 * @return void
 */
function snaps_setup_author() {
	global $wp_query;
	if ( $wp_query->is_author() && isset( $wp_query->post ) ) {
		$GLOBALS['authordata'] = get_userdata( $wp_query->post->post_author );
	}
}
endif;
add_action( 'wp', 'snaps_setup_author' );


if ( ! function_exists( 'snaps_comment' ) ) :
/**
 * Template for comments and pingbacks.
 *
 * Used as a callback by wp_list_comments() for displaying the comments.
 */
function snaps_comment( $comment, $args, $depth ) {
	$GLOBALS['comment'] = $comment;
	if ( 'pingback' == $comment->comment_type || 'trackback' == $comment->comment_type ) :
	?>
		<li class="post pingback">
			<p><?php _e( 'Pingback:', 'snaps' ); ?> <?php comment_author_link(); ?><?php edit_comment_link( __( '(Edit)', 'snaps' ), ' ' ); ?></p>

	<?php
	else :
		global $post; ?>
		<li <?php comment_class(); ?> id="li-comment-<?php comment_ID(); ?>">
			<article id="comment-<?php comment_ID(); ?>" class="comment">
				<footer>
					<div class="comment-author vcard">
						<?php echo get_avatar( $comment, 64 ); ?>
						<?php printf( __( '%s <span class="says">says:</span>', 'snaps' ), sprintf( '<cite class="fn">%s</cite>', get_comment_author_link() ) ); ?>
					</div><!-- .comment-author .vcard -->
					<?php if ( '0' == $comment->comment_approved ) : ?>
						<em><?php _e( 'Your comment is awaiting moderation.', 'snaps' ); ?></em>
						<br />
					<?php endif; ?>

					<div class="comment-meta commentmetadata">
						<a href="<?php echo esc_url( get_comment_link( $comment->comment_ID ) ); ?>"><time datetime="<?php comment_time( 'c' ); ?>">
						<?php
							/* translators: 1: date, 2: time */
							printf( __( '%1$s at %2$s', 'snaps' ), get_comment_date( 'M d, Y' ), get_comment_time() ); ?></time></a>

						<?php edit_comment_link( __( '(Edit)', 'snaps' ), ' ' ); ?>
					</div><!-- .comment-meta .commentmetadata -->
				</footer>

				<div class="comment-content"><?php comment_text(); ?>
					<div class="reply">
						<?php comment_reply_link( array_merge( $args, array( 'depth' => $depth, 'max_depth' => $args['max_depth'] ) ) ); ?>
					</div><!-- .reply -->
				</div>

			</article><!-- #comment-## -->

		<?php
	endif;
}
endif; // ends check for snaps_comment()


/**
 * Return the post URL.
 *
 * @uses get_url_in_content() to get the URL in the post meta (if it exists) or
 * the first link found in the post content.
 *
 * Falls back to the post permalink if no URL is found in the post.
 *
 * @return string The Link format URL.
 */
function snaps_get_link_url() {
	$content = get_the_content();
	$has_url = get_url_in_content( $content );

	return ( $has_url ) ? $has_url : apply_filters( 'the_permalink', get_permalink() );
}


/**
 * Register widgetized area and update sidebar with default widgets
 */
function snaps_widgets_init() {
	register_sidebar( array(
		'name' => __( 'Footer One', 'snaps' ),
		'id' => 'sidebar-1',
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget' => '</aside>',
		'before_title' => '<h1 class="widget-title">',
		'after_title' => '</h1>',
	) );
	register_sidebar( array(
		'name' => __( 'Footer Two', 'snaps' ),
		'id' => 'sidebar-2',
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget' => '</aside>',
		'before_title' => '<h1 class="widget-title">',
		'after_title' => '</h1>',
	) );
	register_sidebar( array(
		'name' => __( 'Footer Three', 'snaps' ),
		'id' => 'sidebar-3',
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget' => '</aside>',
		'before_title' => '<h1 class="widget-title">',
		'after_title' => '</h1>',
	) );
}
add_action( 'widgets_init', 'snaps_widgets_init' );


/**
 * Enqueue scripts and styles
 */
function snaps_scripts() {
	wp_enqueue_style( 'snaps-style', get_stylesheet_uri() );

	wp_enqueue_style( 'genericons', get_template_directory_uri() . '/genericons/genericons.css', array(), '3.3' );

	wp_enqueue_style( 'snaps-lato', snaps_fonts_url(), array(), null );

	wp_enqueue_script( 'snaps-scripts', get_template_directory_uri() . '/js/scripts.js', array( 'jquery' ), '20120206', true );
	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) :
		wp_enqueue_script( 'comment-reply' );
	endif;
}
add_action( 'wp_enqueue_scripts', 'snaps_scripts' );


/**
 * Implement the Custom Header feature
 */
require( get_template_directory() . '/inc/custom-header.php' );

/*
 * Load Jetpack compatibility file
 */
require( get_template_directory() . '/inc/jetpack.php' );

/**
 * Custom template tags for this theme
 */
require( get_template_directory() . '/inc/template-tags.php' );

/**
 * Custom functions that act independently of the theme templates
 */
require( get_template_directory() . '/inc/extras.php' );

/**
 * Functions for the theme customizer
 */
require( get_template_directory() . '/inc/customizer.php' );


// updater for WordPress.com themes
if ( is_admin() )
	include dirname( __FILE__ ) . '/inc/updater.php';
